import os
import tempfile
import shutil
import pysam
from Bio import SeqIO


def parse_alignments(alignments):
    current = None
    for alignment in alignments:
        name = alignment.qname
        if name != current:
            if current:
                yield block
            current = name
            block = []
        block.append(alignment)
    if current:
        yield block

filename = "seqlist.fa"
print("Reading", filename)
stream = open(filename)
records = SeqIO.parse(stream, "fasta")

filename = "seqlist.bam"
print("Reading", filename)
alignments = pysam.AlignmentFile(filename)
blocks = parse_alignments(alignments)

output = tempfile.NamedTemporaryFile(delete=False, mode='wt')
output.close()
temporary_filename = output.name
print("Writing", temporary_filename)
output = pysam.AlignmentFile(temporary_filename, "wb", template=alignments)
for record, block in zip(records, blocks):
    n = len(block)
    name = record.id
    sequence = record.seq
    for i, alignment in enumerate(block):
        assert alignment.qname == name
        if i == 0:
            if alignment.is_unmapped:
                assert not alignment.is_reverse
                alignment.query_sequence = str(sequence)
            elif alignment.is_reverse:
                alignment.query_sequence = str(sequence.reverse_complement())
            else:
                alignment.query_sequence = str(sequence)
            alignment.is_secondary = False
        else:
            alignment.is_secondary = True
        if not alignment.is_unmapped:
            cigartuples = []
            for cigartuple in alignment.cigartuples:
                operation, length = cigartuple
                if operation == pysam.CHARD_CLIP:
                    operation = pysam.CINS
                cigartuple = (operation, length)
                cigartuples.append(cigartuple)
            alignment.cigartuples = cigartuples
        output.write(alignment)
output.close()
stream.close()

print("Moving %s to %s" % (temporary_filename, filename))
shutil.move(temporary_filename, filename)
